/*
  Copyright (c) YourKit GmbH
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  * Neither the name of YourKit nor the
    names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY YOURKIT "AS IS" AND ANY
  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL YOURKIT BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package com.yourkit.probes.builtin;

import com.yourkit.asserts.NotNull;
import com.yourkit.asserts.Nullable;
import com.yourkit.probes.*;

import static com.yourkit.probes.ReflectionUtil.getFieldValue;

public class JPA_DataNucleus {
  private static final class RequestTable extends Table {
    private final StringColumn myMethod = new StringColumn("Method");
    private final StringColumn myDetail = new StringColumn("Detail");

    public RequestTable(@NotNull final String name) {
      super(JPA_DataNucleus.class, name, Table.MASK_FOR_LASTING_EVENTS);
    }
  }
  private static final RequestTable TABLE = new RequestTable(JPA_DataNucleusConstants.TABLE_NAME);

  @Nullable
  private static String getQueryDetail(@NotNull final Object query) {
    final Object queryImpl = getFieldValue(query, "query");
    if (queryImpl == null) {
      return null;
    }
    if (queryImpl.getClass().getName().equals("org.datanucleus.store.rdbms.query.JPQLQuery")) {
      return queryImpl.toString(); // JPQL single-string query
    }
    return getFieldValue(queryImpl, "inputSQL");
  }

  // query

  @MethodPattern({
    "org.datanucleus.api.*.*Query : getResultList() java.util.List",
    "org.datanucleus.api.*.*Query : getSingleResult() Object",
    "org.datanucleus.api.*.*Query : executeUpdate() int"
  })
  public static final class Query {
    public static int onEnter() {
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @This final Object query,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      if (Table.shouldIgnoreRow(row)) {
        // optimization
        return;
      }

      final String queryDetail = getQueryDetail(query);
      final String detail = queryDetail != null ? queryDetail : query.getClass().getName();

      exit(row, methodName, detail, exception);
    }
  }

  // Entity manager methods

  @MethodPattern({
    "org.datanucleus.api.*.*EntityManager : persist(Object) void",
    "org.datanucleus.api.*.*EntityManager : find(Class, Object, *.persistence.LockModeType, java.util.Map)",
    "org.datanucleus.api.*.*EntityManager : close()"
  })
  public static final class EMMethod {
    public static int onEnter() {
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @Param(1) final Object entityOrClass,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      if (Table.shouldIgnoreRow(row)) {
        // optimization
        return;
      }

      exit(
        row,
        methodName,
        entityOrClass != null ? (entityOrClass instanceof Class ? (Class<?>)entityOrClass : entityOrClass.getClass()).getName() : null,
        exception
      );
    }
  }

  // other operations

  @MethodPattern({
    "org.datanucleus.store.rdbms.scostore.*Store : iterator(*) java.util.Iterator"
  })
  public static final class other {
    public static int onEnter() {
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      exit(row, methodName, null, exception);
    }
  }

  // transactions

  @MethodPattern({
    "org.datanucleus.api.*.*EntityTransaction : begin() void",
    "org.datanucleus.api.*.*EntityTransaction : commit() void"
  })
  public static final class Commit {
    public static int onEnter() {
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      exit(row, methodName, null, exception);
    }
  }

  // create entity manager factory (EMF)

  @MethodPattern("org.datanucleus.api.*.PersistenceProviderImpl : createEntityManagerFactory(String, java.util.Map)")
  public static final class createEMF {
    public static int onEnter() {
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @ReturnValue @Nullable final Object createdFactory,
      @Param(1) final String name,
      @ThrownException final Throwable exception
    ) {
      if (Table.shouldIgnoreRow(row)) {
        // optimization
        return;
      }

      exit(
        row,
        "createEntityManagerFactory",
        name + (createdFactory != null ? "" : " (result: null)"),
        exception
      );
    }
  }

  // close entity manager factory (EMF)

  @MethodPattern("org.datanucleus.api.*.*EntityManagerFactory : close()")
  public static final class closeEMF {
    public static int onEnter() {
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @ThrownException final Throwable exception
    ) {
      exit(row, "EntityManagerFactory.close", null, exception);
    }
  }

  // create entity manager (EM)

  @MethodPattern("org.datanucleus.api.*.*EntityManagerFactory : createEntityManager()")
  public static final class CreateEM {
    public static int onEnter() {
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @ReturnValue @Nullable final Object createdFactory,
      @ThrownException final Throwable exception
    ) {
      exit(
        row,
        "createEntityManager",
        createdFactory == null ? "result: null" : null,
        exception
      );
    }
  }

  // common methods

  static int enter() {
    return TABLE.createRow();
  }

  static void exit(
    @OnEnterResult final int row,
    @MethodName final String methodName,
    @Nullable final String detail,
    @Nullable final Throwable exception
  ) {
    TABLE.closeRow(row, exception);
    TABLE.myMethod.setValue(row, methodName);
    TABLE.myDetail.setValue(row, detail);
  }
}
