/*
  Copyright (c) YourKit GmbH
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  * Neither the name of YourKit nor the
    names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY YOURKIT "AS IS" AND ANY
  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL YOURKIT BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package com.yourkit.probes.builtin;

import com.yourkit.asserts.Nullable;
import com.yourkit.probes.*;

import java.nio.file.Path;

public class DirectoryStreams {
  /**
   * Mapping between resource and row index in Directory Stream table
   */
  private static final MasterResourceRegistry<Object> ourStreams = new MasterResourceRegistry<>(
    DirectoryStreams.class,
    DirectoryStreamsConstants.TABLE_NAME,
    "Path"
  );

  @MethodPattern(
    {
      "com.sun.nio.zipfs.ZipDirectoryStream:<init>(*)",
      "sun.nio.fs.*DirectoryStream:<init>(*)"
    }
  )
  @InstanceOf("java.nio.file.DirectoryStream")
  public static final class Init {
    public static long onEnter() {
      return ourStreams.openOnEnter();
    }

    public static void onExit(
      @This final Object stream,
      @Param(1) final Object path,
      @OnEnterResult final long resourceID,
      @ThrownException @Nullable final Throwable e
    ) {
      String pathStr = null;
      if (path instanceof Path) {
        try {
          pathStr = ((Path)path).toUri().getPath();
        }
        catch (final Throwable ignored) {
        }
      }
      ourStreams.openOnExit(resourceID, pathStr, stream, e, FailedEventPolicy.RECORD);
    }
  }

  @MethodPattern(
    {
      "com.sun.nio.zipfs.ZipDirectoryStream:close()",
      "sun.nio.fs.*DirectoryStream:close()",
      "sun.nio.fs.*DirectoryStream:closeImpl()"
    }
  )
  @InstanceOf("java.nio.file.DirectoryStream")
  public static final class Close {
    public static int onEnter(@This final Object stream) {
      return ourStreams.closeOnEnter(stream);
    }

    public static void onExit(
      @OnEnterResult final int row,
      @ThrownException @Nullable final Throwable e
    ) {
      ourStreams.closeOnExit(row, e);
    }
  }
}
