/*
  Copyright (c) 2003-2025 YourKit GmbH
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  * Neither the name of YourKit nor the
    names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY YOURKIT "AS IS" AND ANY
  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL YOURKIT BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package com.yourkit.probes.builtin;

import com.yourkit.asserts.Nullable;
import com.yourkit.probes.*;

import static com.yourkit.probes.ReflectionUtil.callMethod0;
import static com.yourkit.probes.ReflectionUtil.callMethod1;

public final class Servlets {
  private static final class RequestTable extends Table {
    private final StringColumn myURI = new StringColumn("URI");
    private final StringColumn myQuery = new StringColumn("Query");

    public RequestTable() {
      super(Servlets.class, ServletsConstants.TABLE_NAME, Table.MASK_FOR_LASTING_EVENTS);
    }
  }
  private static final RequestTable T_REQUEST = new RequestTable();

  private static final String JakartaHttpServlet = "jakarta.servlet.http.HttpServlet";
  private static final String JavaxHttpServlet = "javax.servlet.http.HttpServlet";

  private static final String JakartaHttpJspPage = "jakarta.servlet.jsp.HttpJspPage";
  private static final String JavaxHttpJspPage = "javax.servlet.jsp.HttpJspPage";

  private static final String JakartaFilter = "jakarta.servlet.Filter";
  private static final String JavaxFilter = "javax.servlet.Filter";

  // Class checkers: index 0 for Jakarta API (first element) and 1 for Javax API (second element)
  private static final ClassChecker HttpServlet_checker = new ClassChecker(JakartaHttpServlet, JavaxHttpServlet);
  private static final ClassChecker HttpJspPage_checker = new ClassChecker(JakartaHttpJspPage, JavaxHttpJspPage);
  private static final ClassChecker Filter_checker = new ClassChecker(JakartaFilter, JavaxFilter);

  private static boolean isJakartaApi(final int mask) {
    return (mask & 1) != 0; // class checker index 0
  }

  @MethodPattern(
    {
      "*:service(jakarta.servlet.http.HttpServletRequest, jakarta.servlet.http.HttpServletResponse) void",
      "*:service(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse) void",
      "-jakarta.servlet.http.HttpServlet:*(*)",
      "-javax.servlet.http.HttpServlet:*(*)",
      "-org.apache.jasper.*:*(*)"
    }
  )
  @RetransformIfInstanceOf({
    JakartaHttpServlet,
    JavaxHttpServlet
  })
  public static final class Servlet_service_Probe {
    public static int onEnter(
      @This final Object _this,
      @Param(1) final Object request
    ) {
      final int mask = HttpServlet_checker.getMask(_this);
      if (mask == 0) {
        return Table.NO_ROW;
      }

      if (HttpJspPage_checker.accepts(_this)) {
        // ignore calls to Servlet's service() for jsp pages
        return Table.NO_ROW;
      }

      return createRequestRow(request, isJakartaApi(mask));
    }

    public static void onExit(
      @OnEnterResult final int rowIndex,
      @ThrownException @Nullable final Throwable e
    ) {
      T_REQUEST.closeRow(rowIndex, e);
    }
  }

  @MethodPattern({
    "*:_jspService(jakarta.servlet.http.HttpServletRequest, jakarta.servlet.http.HttpServletResponse) void",
    "*:_jspService(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse) void"
  })
  @RetransformIfInstanceOf({
    JakartaHttpJspPage,
    JavaxHttpJspPage
  })
  public static final class HttpJspPage_jspService_Probe {
    public static int onEnter(
      @This final Object _this,
      @Param(1) final Object request
    ) {
      final int mask = HttpJspPage_checker.getMask(_this);
      if (mask == 0) {
        return Table.NO_ROW;
      }
      return createRequestRow(request, isJakartaApi(mask));
    }

    public static void onExit(
      @OnEnterResult final int rowIndex,
      @ThrownException @Nullable final Throwable e
    ) {
      T_REQUEST.closeRow(rowIndex, e);
    }
  }

  @MethodPattern({
    "*:doFilter(jakarta.servlet.ServletRequest, jakarta.servlet.ServletResponse, jakarta.servlet.FilterChain)",
    "*:doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)"
  })
  @RetransformIfInstanceOf({
    JakartaFilter,
    JavaxFilter
  })
  public static final class Filter_doFilter_Probe {
    public static int onEnter(
      @This final Object _this,
      @Param(1) final Object request
    ) {
      final int mask = Filter_checker.getMask(_this);
      if (mask == 0) {
        return Table.NO_ROW;
      }
      return createRequestRow(request, isJakartaApi(mask));
    }

    public static void onExit(
      @OnEnterResult final int rowIndex,
      @ThrownException @Nullable final Throwable e
    ) {
      T_REQUEST.closeRow(rowIndex, e);
    }
  }

  private static int createRequestRow(final Object request, final boolean isJakartaApi) {
    final int rowIndex = T_REQUEST.createRow();
    if (Table.shouldIgnoreRow(rowIndex)) {
      // optimization
      return Table.NO_ROW;
    }

    final String uriAttributeName = isJakartaApi ? "jakarta.servlet.include.request_uri" : "javax.servlet.include.request_uri";
    final String includeRequestURI = getRequestAttribute(request, uriAttributeName);

    final String requestURI;
    final String query;
    if (includeRequestURI == null || includeRequestURI.isEmpty()) {
      requestURI = getRequestURI(request);
      query = getQuery(request);
    }
    else {
      requestURI = includeRequestURI;
      final String queryAttributeName = isJakartaApi ? "jakarta.servlet.include.query_string" : "javax.servlet.include.query_string";
      query = getRequestAttribute(request, queryAttributeName);
    }

    T_REQUEST.myURI.setValue(rowIndex, requestURI);
    T_REQUEST.myQuery.setValue(rowIndex, query);

    return rowIndex;
  }

  /**
   * This method uses reflection to avoid class-loading issues
   */
  private static String getQuery(final Object request) {
    if (request == null) {
      return "<unknown>";
    }

    final CallStatus callStatus = new CallStatus();
    final String result = (String)callMethod0(request, "getQueryString", callStatus);
    if (callStatus.failed()) {
      return getQuery(tryGetWrappedRequest(request));
    }

    return result;
  }

  /**
   * This method uses reflection to avoid class-loading issues
   */
  static String getRequestURI(final Object request) {
    if (request == null) {
      return "<unknown>";
    }

    final CallStatus callStatus = new CallStatus();
    final String result = (String)callMethod0(request, "getRequestURI", callStatus);
    if (callStatus.failed()) {
      return getRequestURI(tryGetWrappedRequest(request));
    }

    return result;
  }

  /**
   * This method uses reflection to avoid class-loading issues
   */
  private static String getRequestAttribute(final Object request, final String attributeName) {
    if (request == null) {
      return "<unknown>";
    }

    final CallStatus callStatus = new CallStatus();
    final Object resultObj = callMethod1(
      request.getClass(), request, "getAttribute:(Ljava/lang/String;)Ljava/lang/Object;", attributeName,
      callStatus
    );
    if (callStatus.failed()) {
      return getRequestAttribute(tryGetWrappedRequest(request), attributeName);
    }

    if (!(resultObj instanceof String)) {
      return null;
    }

    final String result = (String)resultObj;
    if (result.startsWith("//")) {
      return result.substring(1);
    }
    return result;
  }

  /**
   * This method uses reflection to avoid class-loading issues
   */
  private static Object tryGetWrappedRequest(final Object requestWrapper) {
    return callMethod0(requestWrapper, "getRequest");
  }
}
