/*
  Copyright (c) 2003-2025 YourKit GmbH
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  * Neither the name of YourKit nor the
    names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY YOURKIT "AS IS" AND ANY
  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL YOURKIT BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package com.yourkit.probes.builtin;

import com.yourkit.asserts.NotNull;
import com.yourkit.asserts.Nullable;
import com.yourkit.probes.*;

import static com.yourkit.probes.ReflectionUtil.callMethod0;
import static com.yourkit.probes.ReflectionUtil.getFieldValue;

public class JPA_EclipseLink {
  private static final class RequestTable extends Table {
    private final StringColumn myMethod = new StringColumn("Method");
    private final StringColumn myDetail = new StringColumn("Detail");

    public RequestTable(@NotNull final String name) {
      super(JPA_EclipseLink.class, name, Table.MASK_FOR_LASTING_EVENTS);
    }
  }
  private static final RequestTable TABLE = new RequestTable(JPA_EclipseLinkConstants.TABLE_NAME);

  @Nullable
  private static String getQueryDetail(@NotNull final Object query) {
    final String queryName = getFieldValue(query, "queryName");

    String jpqlOrSql = null;
    final Object databaseQuery = getFieldValue(query, "databaseQuery");
    if (databaseQuery != null) {
      jpqlOrSql = (String)callMethod0(databaseQuery, "getJPQLString");
      if (jpqlOrSql == null) {
        jpqlOrSql = (String)callMethod0(databaseQuery, "getSQLString");
      }
    }

    return composeString(queryName, " = ", jpqlOrSql);
  }

  // query

  @MethodPattern({
    "org.eclipse.persistence.*Query*:getResultList() java.util.List",
    "org.eclipse.persistence.*Query*:getSingleResult() Object",
    "org.eclipse.persistence.*Query*:executeUpdate() int"
  })
  public static final class Query {
    public static int onEnter() {
      incCallLevel();

      if (getCallLevel() > 1) {
        return Table.NO_ROW;
      }

      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @This final Object query,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      decCallLevel();

      if (Table.shouldIgnoreRow(row)) {
        // optimization
        return;
      }

      final String queryDetail = getQueryDetail(query);
      final String detail = queryDetail != null ? queryDetail : query.getClass().getName();

      exit(row, methodName, detail, exception);
    }
  }

  // Entity manager methods

  @MethodPattern({
    "org.eclipse.persistence.internal.jpa.EntityManagerImpl : persist(Object) void",
    "org.eclipse.persistence.internal.jpa.EntityManagerImpl : find(Class, Object, jakarta.persistence.LockModeType, java.util.Map)",
    "org.eclipse.persistence.internal.jpa.EntityManagerImpl : find(Class, Object, javax.persistence.LockModeType, java.util.Map)",
    "org.eclipse.persistence.internal.jpa.EntityManagerImpl : close()"
  })
  public static final class EMMethod {
    public static int onEnter() {
      incCallLevel();

      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @Param(1) final Object entityOrClass,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      decCallLevel();

      if (Table.shouldIgnoreRow(row)) {
        // optimization
        return;
      }

      exit(
        row,
        methodName,
        entityOrClass != null ? (entityOrClass instanceof Class ? (Class<?>)entityOrClass : entityOrClass.getClass()).getName() : null,
        exception
      );
    }
  }

  // other operations

  @MethodPattern({
    "org.eclipse.persistence.internal.databaseaccess.DatabaseAccessor : basicBeginTransaction(*)",
    "org.eclipse.persistence.internal.databaseaccess.DatabaseAccessor : basicCommitTransaction(*)",
    "org.eclipse.persistence.internal.databaseaccess.DatabaseAccessor : basicExecuteCall(*)",
    "org.eclipse.persistence.internal.databaseaccess.DatabaseAccessor : basicRollbackTransaction(*)"
  })
  public static final class other {
    public static int onEnter() {
      if (getCallLevel() > 0) {
        return Table.NO_ROW;
      }
      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      exit(row, methodName, null, exception);
    }
  }

  // transactions

  @MethodPattern({
    "org.eclipse.persistence.internal.jpa.transaction.EntityTransactionImpl : begin() void",
    "org.eclipse.persistence.internal.jpa.transaction.EntityTransactionImpl : commit() void"
  })
  public static final class Commit {
    public static int onEnter() {
      incCallLevel();

      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @MethodName final String methodName,
      @ThrownException final Throwable exception
    ) {
      decCallLevel();

      exit(row, methodName, null, exception);
    }
  }

  // create entity manager factory (EMF)

  @MethodPattern("org.eclipse.persistence.jpa.PersistenceProvider : createEntityManagerFactory(String, java.util.Map)")
  public static final class createEMF {
    public static int onEnter() {
      incCallLevel();

      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @ReturnValue @Nullable final Object createdFactory,
      @Param(1) final String name,
      @ThrownException final Throwable exception
    ) {
      decCallLevel();

      if (Table.shouldIgnoreRow(row)) {
        // optimization
        return;
      }

      exit(
        row,
        "createEntityManagerFactory",
        name + (createdFactory != null ? "" : " (result: null)"),
        exception
      );
    }
  }

  // close entity manager factory (EMF)

  @MethodPattern("org.eclipse.persistence.internal.jpa.EntityManagerFactoryDelegate : close()")
  public static final class closeEMF {
    public static int onEnter() {
      incCallLevel();

      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @ThrownException final Throwable exception
    ) {
      decCallLevel();

      exit(row, "EntityManagerFactory.close", null, exception);
    }
  }

  // create entity manager (EM)

  @MethodPattern({
    "org.eclipse.persistence.internal.jpa.EntityManagerFactoryDelegate : createEntityManagerImpl(java.util.Map, jakarta.persistence.SynchronizationType)",
    "org.eclipse.persistence.internal.jpa.EntityManagerFactoryDelegate : createEntityManagerImpl(java.util.Map, javax.persistence.SynchronizationType)"
  })
  public static final class CreateEM {
    public static int onEnter() {
      incCallLevel();

      return enter();
    }

    public static void onExit(
      @OnEnterResult final int row,
      @ReturnValue @Nullable final Object createdFactory,
      @ThrownException final Throwable exception
    ) {
      decCallLevel();

      exit(
        row,
        "createEntityManager",
        createdFactory == null ? "result: null" : null,
        exception
      );
    }
  }

  // common methods

  static int enter() {
    return TABLE.createRow();
  }

  static void exit(
    @OnEnterResult final int row,
    @MethodName final String methodName,
    @Nullable final String detail,
    @Nullable final Throwable exception
  ) {
    TABLE.closeRow(row, exception);
    TABLE.myMethod.setValue(row, methodName);
    TABLE.myDetail.setValue(row, detail);
  }

  @Nullable
  private static String composeString(@Nullable final String s1, final String separator, @Nullable final String s2) {
    if (s1 == null && s2 == null) {
      return null;
    }

    return
      (s1 != null ? s1 : "") +
      (s1 != null && s2 != null ? separator : "") +
      (s2 != null ? s2 : "");
  }

  private static final ThreadLocal<int[]> ourCallLevelCount =
    ThreadLocal.withInitial(() -> new int[1]);

  private static void decCallLevel() {
    ourCallLevelCount.get()[0]--;
  }

  private static void incCallLevel() {
    ourCallLevelCount.get()[0]++;
  }

  private static int getCallLevel() {
    return ourCallLevelCount.get()[0];
  }
}
